% 
% Copyright (c) 2008 Kristopher L. Kuhlman (kuhlman at hwr dot arizona dot edu)
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in
% all copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
% THE SOFTWARE.
% 

% Kuhlman, K.L. and A.W. Warrick, 2008. Quasilinear infiltration from 
% an elliptical cavity, Advances in Water Resources, 31(8), 1057–1065. 
% http://dx.doi.org/10.1016/j.advwatres.2008.04.009
%
% this Matlab script computes the Mathieu functions required for the
% elliptical cutout problem

% $Id: elliptical_cutout_vert.m,v 1.11 2008/04/07 22:05:23 kris Exp kris $

%% set these parameters
elA = 1.0;
ecc = 0.9;
N = 20;          % number of terms in generalized Fourier series expansion

%% computed parameters
F = elA*ecc;        % dimensionless semi-focal distance (*alpha/2)
e0 = acosh(1/ecc);      % elliptical radius of cutout
elB = F*sinh(e0);   % dim-less semi-minor dist

% compute solution on an elliptical mesh 
eta = linspace(e0,10.0,50);   % eta values to compute solution at
psi = linspace(-pi,pi,100);      % psi values "   "    "  "

ne = length(eta);
np = length(psi);
R = max(2*N+12,40);        % size of "infinite" matrix
q = (F*F)/4.0;           % Mathieu parameter (remove negative from definition)

%% setup and compute eigenvalue problem
MEv = diag((2*(0:R-1)').^2,0) + diag(ones(R-1,1)*q,-1) + diag(ones(R-1,1)*q,+1);
MEv(2,1) = 2.0*q;
[AEv,LEv] = eig(MEv);

MOd = diag((2*(0:R-1)' + 1).^2,0) + diag(ones(R-1,1)*q,-1) + diag(ones(R-1,1)*q,+1);
MOd(1,1) = 1.0 + q;
[BOd,LOd] = eig(MOd);

% set sign of eigenvalues
for col = 1:R
  if AEv(col,col) < 0.0
    AEv(:,col) = -AEv(:,col);
  end
  if BOd(col,col) < 0.0
    BOd(:,col) = -BOd(:,col);
  end
end


%% normalize eigenvectors (zero term included twice)
% other eigenvalues already normalized to unit length by LAPACK
norm = 2*AEv(1,1)^2 + sum(AEv(2:R,1).^2);
AEv(1:R,1) = AEv(1:R,1)/sqrt(norm);

%% compute cosine-elliptic (ce), for each value of psi
ceEv = zeros(N,np);  ceOd = zeros(N,np);
iv(1:R,1) = (-1.0).^(0:R-1).';

% MF are indexed by n, rather than 2n or 2n+1
% since only even ce and odd se are used
for p = 1:np
  for n = 1:N
    ceEv(n,p) = iv(n)*sum(iv(1:R).*AEv(1:R,n).*cos( 2*(0:R-1)'*psi(p)));
    ceOd(n,p) = iv(n)*sum(iv(1:R).*BOd(1:R,n).*cos((2*(0:R-1)'+1)*psi(p)));
  end
end

%% compute Ke and Ko for each value of eta
KeEv = zeros(N,ne);    KeOd = zeros(N,ne);
besi = zeros(R+1,ne);  besk = zeros(R+1,ne);
v = zeros(2,ne);

expeta(1,1:ne) = exp(-eta(1:ne));
expeta(2,1:ne) = exp( eta(1:ne));
v(1,1:ne) = sqrt(q)*expeta(1,1:ne);
v(2,1:ne) = sqrt(q)*expeta(2,1:ne);

for e = 1:ne
  besi(1:R+1,e) = besseli(0:R, v(1,e))';
  besk(1:R+1,e) = besselk(0:R, v(2,e))';
end

% <<< these are not actually radial MF, the part that       >>>
% <<< cancels when forming the ratio of two functions       >>>
% <<< is left out, since they are only ever used as a ratio.>>>
for n = 1:N
  KeEv(n,1:ne) = sum(AEv(1:R,n*ones(1,ne)).*besi(1:R,1:ne).*besk(1:R,1:ne),1);
  KeOd(n,1:ne) = sum(BOd(1:R,n*ones(1,ne)).* ...
                     (besi(1:R,1:ne).*besk(2:R+1,1:ne) - besi(2:R+1,1:ne).*besk(1:R,1:ne)),1);
end

%% compute Ke' for each value of eta
DKeEv = zeros(N,ne);    DKeOd = zeros(N,ne); 
Dbesi = zeros(R+1,ne);  Dbesk = zeros(R+1,ne);

% use recurrences to compute BF derivs
Dbesi(1,1) =  besi(2,1);
Dbesk(1,1) = -besk(2,1);
Dbesi(2:R,1) =  0.5*(besi(1:R-1,1) + besi(3:R+1,1));
Dbesk(2:R,1) = -0.5*(besk(1:R-1,1) + besk(3:R+1,1));
Dbesi(R+1,1) =  besi(R,1) - (R+1)/v(1,1).*besi(R+1,1);
Dbesk(R+1,1) = -besk(R,1) - (R+1)/v(2,1).*besk(R+1,1);

for n=1:N
  DKeEv(n,1) = sqrt(q)*sum(AEv(1:R,n).* ...
    (besi(1:R,1).*Dbesk(1:R,1).*expeta(2*ones(R,1),1) - ...
    Dbesi(1:R,1).*besk(1:R,1).* expeta(1*ones(R,1),1)),1);
  DKeOd(n,1) = sqrt(q)*sum(BOd(1:R,n).* ...
    (besi(1:R,1).*Dbesk(2:R+1,1).*expeta(2*ones(R,1),1) - ...
    Dbesi(1:R,1).*besk(2:R+1,1).*expeta(1*ones(R,1),1) - ...
    besi(2:R+1,1).*Dbesk(1:R,1).*expeta(2*ones(R,1),1) + ...
    Dbesi(2:R+1,1).*besk(1:R,1).*expeta(1*ones(R,1),1)),1);
end

%% compute H for all points on grid
besiREv(1:R,1) = besseli(2*(0:R-1),  elA).';
besiROd(1:R,1) = besseli(2*(0:R-1)+1,elA).';

% additional (-1)^r term in vertical ellipse, compared to horiz. ellipse
coeffEv(1:N,1) =  2.0*iv(1:N).*sum(iv(1:R,ones(1,N)).*AEv(1:R,1:N).*besiREv(1:R,ones(1,N)),1).';
coeffOd(1:N,1) = -2.0*iv(1:N).*sum(iv(1:R,ones(1,N)).*BOd(1:R,1:N).*besiROd(1:R,ones(1,N)),1).';

H = zeros(ne,np);
for e=1:ne
  for p=1:np
    H(e,p) = sum(coeffEv(1:N).*ceEv(1:N,p).*KeEv(1:N,e)./KeEv(1:N,1) + ...
                 coeffOd(1:N).*ceOd(1:N,p).*KeOd(1:N,e)./KeOd(1:N,1));
  end
end

% HCheck = zeros(N,np);
% for n = 1:N
%   for p = 1:np
%     HCheck(n,p) = sum(coeffEv(1:n).*ceEv(1:n,p) + coeffOd(1:n).*ceOd(1:n,p));
%   end
% end

%% compute dimensionless flux along the boundary of ellipse
% as a function of angular position
V0 = zeros(np,1);

for p=1:np
  flux = exp(elA*cos(psi(p)))*(...
      sum(coeffEv(1:N).*ceEv(1:N,p).*DKeEv(1:N,1)./KeEv(1:N,1) + ...
          coeffOd(1:N).*ceOd(1:N,p).*DKeOd(1:N,1)./KeOd(1:N,1)) + ...
      elB*cos(psi(p))*sum(coeffEv(1:N).*ceEv(1:N,p) + coeffOd(1:N).*ceOd(1:N,p)));
  V0(p) = 2.0*elB*cos(psi(p)) - flux;
end

%% compute things for plotting

X = F*cosh(eta(ones(np,1),1:ne))'.*cos(psi(ones(ne,1),1:np)); % x is positive down
Z = F*sinh(eta(ones(np,1),1:ne))'.*sin(psi(ones(ne,1),1:np)); 
phi = 0.5*log(H.*exp(X));
PHI = phi - X;
if F <= 0.1
  phiC = 0.5*log(HC.*exp(X));
  PHIC = phiC - X;
end

%% plot figures
range = [0,4,-1,6];
hcontours = -10:0.5:5;
pcontours = -10:0.25:5;

% hydraulic head contours
figure
[C,h] = contour(Z,X,PHI,hcontours);
set(gca,'YDir','reverse')
% xlabel('Z','FontSize',15)
% ylabel('X','FontSize',15)
daspect([1 1 1]);
axis(range);
hold on;
colormap(zeros(size(gray)))
plot(elB*sin(psi),elA*cos(psi),'-r','LineWidth',4);
clabel(C,h,'manual','FontSize',15,'Rotation',0)
h(1) = text('Position',[1.9,6.5],'fontsize',15, ...
  'interpreter','latex','string','$\tilde{Z}$');
h(2) = text('Position',[-0.5,2.5],'fontsize',15, ...
  'interpreter','latex','string','$\tilde{X}$');

% pressure head contours   
figure
[C,h] = contour(Z,X,phi,pcontours);
set(gca,'YDir','reverse')
% xlabel('Z','FontSize',15)
% ylabel('X','FontSize',15)
daspect([1 1 1]);
axis(range);
hold on;
colormap(zeros(size(gray)))
plot(elB*sin(psi),elA*cos(psi),'-r','LineWidth',4);
clabel(C,h,'manual','FontSize',15,'Rotation',0)
h(3) = text('Position',[1.9,6.5],'fontsize',15, ...
  'interpreter','latex','string','$\tilde{Z}$');
h(4) = text('Position',[-0.5,2.5],'fontsize',15, ...
  'interpreter','latex','string','$\tilde{X}$');



