!
! Copyright (c) 2011 Kristopher L. Kuhlman (klkuhlm at sandia dot gov)
! 
! Permission is hereby granted, free of charge, to any person obtaining a copy
! of this software and associated documentation files (the "Software"), to deal
! in the Software without restriction, including without limitation the rights
! to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
! copies of the Software, and to permit persons to whom the Software is
! furnished to do so, subject to the following conditions:
! 
! The above copyright notice and this permission notice shall be included in
! all copies or substantial portions of the Software.
! 
! THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
! IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
! FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
! AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
! LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
! OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
! THE SOFTWARE.
!

!! this file has _two_ modules in it 

!! ######################################################################
!! ######################################################################

module shared_data
  use constants, only : DP
  use inverse_Laplace_transform, only : INVLT
  implicit none

  ! variables to pass using modules

  ! parameters to change between calculations
  real(DP), save :: tsval  ! time - passed through hankel integral to Lap xform

  ! aquifer parameters (secondary - computed from above)
  real(DP), save :: bD, lD, dD, rDw, CD
  real(DP), save :: alphaD, gamma, kappa
  real(DP), save, dimension(2) :: beta

  type(INVLT), save :: lap

end module shared_data

!! ######################################################################
!! ######################################################################

module utilities
implicit none
private
public :: ccosh, csinh, logspace

contains
  pure elemental function ccosh(z) result(f)
    use constants, only : EP
    complex(EP), intent(in) :: z
    complex(EP) :: f
    real(EP) :: x,y
    x = real(z)
    y = aimag(z)
    f = cmplx(cosh(x)*cos(y), sinh(x)*sin(y),EP)
  end function ccosh
  
  pure elemental function csinh(z) result(f)
    use constants, only : EP
    complex(EP), intent(in) :: z
    complex(EP) :: f
    real(EP) :: x,y
    x = real(z)
    y = aimag(z)
    f = cmplx(sinh(x)*cos(y), cosh(x)*sin(y),EP)
  end function csinh
  
  pure function linspace(lo,hi,num) result(v)
    use constants, only : DP
    real(DP), intent(in) :: lo,hi
    integer, intent(in) :: num
    real(DP), dimension(num) :: v
    integer :: i
    real(DP) :: rnum, range, sgn

    rnum = real(num - 1,DP)
    range = abs(hi - lo) 
    sgn = sign(1.0_DP,hi-lo) ! if lo > high, count backwards
    forall (i=0:num-1) v(i+1) = lo + sgn*real(i,DP)*range/rnum
  end function linspace

  pure function logspace(lo,hi,num) result(v)
    use constants, only : DP
    integer, intent(in) :: lo,hi,num
    real(DP), dimension(num) :: v
    v = 10.0_DP**linspace(real(lo,DP),real(hi,DP),num)
  end function logspace

end module utilities


